/* $Id: zap.c,v 1.63 1999/04/23 15:29:25 dhiller Exp $ */
/* Copyright (C) 1994 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Don Hiller, with guidance from install.c */

#include "sema.h"
#if !defined(E1485_SOURCE) && !defined(HPVXI_DOWNLOAD)
#include <fcntl.h>
#endif

#ifdef	MICROSOFT_OS
#  include <io.h>
#  ifdef __STDC__       /* non-ansi names used in this file */
#    define open _open
#    define close _close
#    define lseek _lseek
#    define read _read
#  endif
#else
#  if !defined(E1485_SOURCE) && !defined(HPVXI_DOWNLOAD)
#    include <unistd.h>
#  endif
#  define O_BINARY	0
#endif

/* This is the EEPROM I/O 56307 downloadable for Sonata B. */
static ULONGSIZ32 son_ee_code[] =
{
#include "son_ee.asc"
};
static int son_ee_code_size = sizeof(son_ee_code)/sizeof(ULONGSIZ32);



#define ERASE_BLOCK     0x20
#define CONFIRM_ERASE   0xD0
#define READ_MODE       0xFF
#define PROG_MODE       0x40
#define CHECK_STATUS    0x70
#define CLEAR_STATUS    0x50
#define GOOD_STATUS     0x80
#define STATUS_MASK     0xF8
#define READ_ID         0x90
#define FPROM_DEVICE_CODE 0x95
#define BOOT_START	0x0040000
#define ID_CAL_START	0x0048000
#define PARM2_START	0x004c000
#define VPP_REG		0x8014
#define VPP_ON		0x08
#define VPP_OFF		0x00
#define ERASE_SUSP_BIT  0x40
#define ERASE_ERROR_BIT 0x20
#define PROG_ERROR_BIT  0x10
#define VPP_LOW_BIT     0x08
#define NUM_STAT_CHK    10

/* Normal downloadable command vectors for Sonata B */
#define  E1433_SCA_SONATA_ECHO			0x0800
/* Command vectors for Sonata B EEPROM I/O downloadable in  son_ee.asc */
#define E1433_SCA_SONATA_EE_READ_ERRS		0x8600
#define E1433_SCA_SONATA_EE_ADDR		0x8700
#define E1433_SCA_SONATA_EE_DATA		0x8800
#define E1433_SCA_SONATA_EE_READ		0x8900
#define E1433_SCA_SONATA_EE_STAT		0x8a00
#define E1433_SCA_SONATA_EE_WRITE_EN		0x8b00
#define E1433_SCA_SONATA_EE_WRITE_DIS		0x8c00
#define E1433_SCA_SONATA_EE_WRITE		0x8d00
#define E1433_SCA_SONATA_EE_ERASE_ALL		0x8e00
#define E1433_SCA_SONATA_EE_WR_STAT		0x8f00

#define CHK(func) { error = (func); if (error) goto cleanup; }
#define ERROROUT(pstr) { PRINTF(pstr); error = ERR1432_BAD_FLASH; goto cleanup; }

#define VALID_PARM2_REV(r)	\
    ( (r) >= E1432_PARM2_R1 && (r) <= E1432_PARM2_R_CURR )

/* revs > 2 program first rev length with E1432_ID_CAL_R2_REV_LENGTH */
#define VALID_ID_CAL_REV(r)	( (r) == E1432_ID_CAL_R2_REV_LENGTH )

#define VALID_OPT_CHAR(c)	\
    ( (c) == '\0' || ((c) >= 'A' && (c) <= 'Z') || ((c) >= '0' && (c) <= '9') )
#define	VALID_OPT_STR_LEN_MAX	4
#define	VALID_OPT_STR_LEN_MIN	3

#if 0
/***************************************************************************
 Print out a block of long data in hex.
 **************************************************************************/
static void
print_block(LONGSIZ32 size, LONGSIZ32 *buff, char* msg)
{
    LONGSIZ32 *lptr = buff;
    int i;
    (void) PRINTF("%s\n", msg);
    while ( lptr - buff < size )
    {
        for ( i = 0; i < 8 && lptr - buff < size; i++ )
            (void) PRINTF(" %.8lx", *lptr++);
        (void) PRINTF("\n");
    }
}
#endif

/***************************************************************************
 Substrate F-PROM code
 **************************************************************************/

#if 0

/***************************************************************************
 Prints an ID/CAL block to stdout.
 **************************************************************************/
static void
print_id_cal(LONGSIZ32 *id_cal_blk_ptr)
{
    E1432_ID_CAL_R2* id_cal_ptr = (E1432_ID_CAL_R2*) id_cal_blk_ptr;
    const int date_size = sizeof(E1432_COMPRESSED_DATE);
    char format[100];

    (void) PRINTF("id_cal = 0x%lx {\n",id_cal_ptr);
    (void) PRINTF("    rev_length = 0x%lx\n",id_cal_ptr->rev_length);
    (void) sprintf(format, "    serial = \"%%.%ds\"\n", E1432_SERIAL_LENGTH);
    (void) PRINTF(format, (char*)&id_cal_ptr->serial[0]);
    (void) PRINTF("    dc_cal = {\n");
    (void) sprintf(format, "        cal time  = %%.%ds\n", date_size);
	(void) PRINTF(format, (char*)&id_cal_ptr->dc_cal.date);
        (void) PRINTF("        lo_range = {");
            (void) PRINTF(" full_scale = %g",id_cal_ptr->dc_cal.lo_range.full_scale);
            (void) PRINTF(", offset = %g } \n",id_cal_ptr->dc_cal.lo_range.offset);
        (void) PRINTF("        hi_range = {");
            (void) PRINTF(" full_scale = %g",id_cal_ptr->dc_cal.hi_range.full_scale);
            (void) PRINTF(", offset = %g }\n",id_cal_ptr->dc_cal.hi_range.offset);
        (void) PRINTF("        gnd_offset = %g\n",id_cal_ptr->dc_cal.gnd_offset);
    (void) PRINTF("    }\n");
    (void) PRINTF("    board_date_code = %d\n", id_cal_ptr->board_date_code);

}
#endif


/***************************************************************************
 Turns off the programming voltage
 **************************************************************************/
static int turnOffVpp(E1432_MODULE_LIST_NODE *mn)
{
    int error;
    SHORTSIZ16 page;

    TRACE_PRINTF(2, ("e1432_zap() turning off Vpp\n"));

    page = E1432_VPP_PAGE;
    if (mn->a24_256k)
	page <<= 2;
    error = i1432_direct_write_register(mn, E1432_PAGE_MAP_REG,
					page);
    if ( error != 0 ) return error;

    error = i1432_direct_write32_register_a(mn, VPP_REG, E1432_A24, VPP_OFF);

    return error;
}


/***************************************************************************
 Checks for good F-PROM status
 **************************************************************************/
static int
status_ok(E1432_MODULE_LIST_NODE *mn, LONGSIZ32 addr, LONGSIZ32 *status)
{
    LONGSIZ32 i;

    for ( i = 0; i < NUM_STAT_CHK; i++ )
    {
	(void) i1432_direct_write32_register_a(mn, addr, E1432_A24,
	  CHECK_STATUS);
	(void) i1432_direct_read32_register_a(mn, addr, E1432_A24, status);
    	*status &= STATUS_MASK;
    	if ( *status == GOOD_STATUS ) return 1;
    }
    
    return 0;  
}

/***************************************************************************
 Performs the FLASH PROM programming in the substrate

 parms:
   mn = module pointer
   prom_addr = absolute address of F-PROM segment to be programmed
   size = length of block to be programmed, in bytes
   buf = pointer to block in memory to be programmed into F-PROM
 **************************************************************************/
SHORTSIZ16
i1432_subst_zap(E1432_MODULE_LIST_NODE *mn, ULONGSIZ32 prom_addr,
 LONGSIZ32 size, LONGSIZ32 *buff)
{
    LONGSIZ32   status;
    ULONGSIZ32 addr;
    SHORTSIZ16 error = 0;
    int vpp_on = 0;  /* keep track of state of programming voltage */
    LONGSIZ32 fprom_id;
    LONGSIZ32* buff_ptr = buff;
    int i, j;
    LONGSIZ32 prog_word;
    LONGSIZ32 temp;
    SHORTSIZ16 page;

    TRACE_PRINTF(0, ("i1432_subst_zap(0x%p, 0x%lx, 0x%lx, 0x%p)\n",
		     mn, prom_addr, size, buff));

    /* program page register */
    page = E1432_FPROM_PAGE;
    if (mn->a24_256k)
    {
	page <<= 2;
	page += 2;
    }
    CHK(i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, page));

    /* check id of F-PROM to make sure we can program it */
    CHK(i1432_direct_write32_register_a(mn, prom_addr, E1432_A24, READ_ID));
    /* read and discard Manufactuer ID so this will work with all vendors */
    /* upper byte */
    CHK(i1432_direct_read32_register_a(mn, prom_addr, E1432_A24, &temp));
    /* read and check device ID */
    /* lower byte */
    CHK(i1432_direct_read32_register_a(mn, prom_addr+4, E1432_A24, &temp));
    fprom_id = temp & 0xFF;
    if (fprom_id != FPROM_DEVICE_CODE)
    {
        i1432_error_info = "Incorrect ROM device code.";
	error = i1432_la_print_error(mn->la, ERR1432_BAD_FLASH);
	ERROROUT(("i1432_subst_zap(): Wrong flash PROM DEVICE CODE,"
	 " expected 0x%X, got 0x%x\n", (int)FPROM_DEVICE_CODE, (int)fprom_id))
    }
    /* (void) PRINTF("flash PROM ID = 0x%x\n",fprom_id); */

    /* clear status, check for good status */
    TRACE_PRINTF(1, ("e1432_zap() clearing F-PROM status,"
     " checking for good status\n"));
    CHK(i1432_direct_write32_register_a(mn, prom_addr, E1432_A24,
      CLEAR_STATUS));
    if ( ! status_ok(mn, prom_addr, &status) )
    {
        i1432_error_info = "Bad ROM status.";
	error = i1432_la_print_error(mn->la, ERR1432_BAD_FLASH);
	ERROROUT(("i1432_subst_zap(): Bad FPROM status,"
	 " expected 0x%X, got 0x%x\n", (int)GOOD_STATUS, (int)status))
    }

    /* turn on programming voltage */
    TRACE_PRINTF(1, ("e1432_zap() turning on Vpp\n"));
    page = E1432_VPP_PAGE;
    if (mn->a24_256k)
	page <<= 2;
    CHK(i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, page));
    vpp_on = 1;  /* remember that we have Vpp turned on */
    CHK(i1432_direct_write32_register_a(mn, VPP_REG, E1432_A24, VPP_ON));
    i1432_pause(1.0);
    /* restore page map register to F-PROM space */
    page = E1432_FPROM_PAGE;
    if (mn->a24_256k)
    {
	page <<= 2;
	page += 2;
    }
    CHK(i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, page));

    /* erase block */
    TRACE_PRINTF(1, ("e1432_zap() erasing block\n"));
    CHK(i1432_direct_write32_register_a(mn, prom_addr, E1432_A24, ERASE_BLOCK));
    CHK(i1432_direct_write32_register_a(mn, prom_addr, E1432_A24,
      CONFIRM_ERASE));
    i1432_pause(3);
    if ( ! status_ok(mn, prom_addr, &status) ) i1432_pause(3);
    if ( ! status_ok(mn, prom_addr, &status) )
    {
	status &= STATUS_MASK;
	if ( status & VPP_LOW_BIT )
	{
            i1432_error_info = "Low ROM Vpp.";
	    error = i1432_la_print_error(mn->la, ERR1432_VPP_ERROR);
	    ERROROUT(("i1432_subst_zap(): FPROM erase failed due to low VPP."))
	}
	else if ( status & ERASE_ERROR_BIT )
	{
            i1432_error_info = "ROM Erase error.";
	    error = i1432_la_print_error(mn->la, ERR1432_ROM_BLOCK_ERASE);
	    ERROROUT(("i1432_subst_zap():"
	     " FPROM erase failed due to erase error."))
	}
	else if ( status & PROG_ERROR_BIT )
	{
            i1432_error_info = "ROM Programming error.";
	    error = i1432_la_print_error(mn->la, ERR1432_BAD_FLASH);
	    ERROROUT((
	     "i1432_subst_zap(): FPROM erase failed due to programing error."))
	}
	else if ( status & ERASE_SUSP_BIT )
	{
            i1432_error_info = "ROM Erase suspend error.";
	    error = i1432_la_print_error(mn->la, ERR1432_BAD_FLASH);
	    ERROROUT((
	     "i1432_subst_zap(): FPROM erase failed due to erase suspending."))
	}
    }
    CHK(i1432_direct_write32_register_a(mn, prom_addr, E1432_A24,
      CLEAR_STATUS));

    /* check that erase took */
    TRACE_PRINTF(1, ("e1432_zap() checking that erase took\n"));
    CHK(i1432_direct_write32_register_a(mn, prom_addr, E1432_A24, READ_MODE));
    CHK(i1432_direct_read32_register_a(mn, prom_addr, E1432_A24, &temp));
    if ( (temp & 0xff) != 0xff )
    {
        i1432_error_info = "Erase failed.";
	error = i1432_la_print_error(mn->la, ERR1432_ROM_BLOCK_ERASE);
        ERROROUT(("i1432_subst_zap() failed to erase block."))
    }

    /* do the actual programming*/
    TRACE_PRINTF(1, ("e1432_zap() programming\n"));
    addr = prom_addr;
    for ( i = 0; i < size; i += 4 )
    {
        prog_word = *buff_ptr;
	for ( j = 0; j < 4; j++ )
	{
            CHK(i1432_direct_write32_register_a(mn, addr, E1432_A24,
	      PROG_MODE));
            CHK(i1432_direct_write32_register_a(mn, addr, E1432_A24,
	      prog_word));
            if ( ! status_ok(mn, addr, &status) )
            {
                i1432_error_info = "Bad ROM status.";
	        error = i1432_la_print_error(mn->la, ERR1432_BAD_FLASH);
	        ERROROUT(("i1432_subst_zap():"
		 " Bad FPROM status during programming,"
	         " expected 0x%X, got 0x%x.\n", (int)GOOD_STATUS, (int)status))
            }
            CHK(i1432_direct_write32_register_a(mn, addr, E1432_A24,
	      READ_MODE));
            CHK(i1432_direct_read32_register_a(mn, addr, E1432_A24,
	      &temp));
	    if ( (temp & 0xff) != (prog_word & 0xff) )
	    {
                i1432_error_info = "Bad ROM data.";
	        error = i1432_la_print_error(mn->la, ERR1432_BAD_FLASH);
	        ERROROUT(("e1432_zap():"
		 " programmed 0x%x, read back 0x%x, at word %d.\n",
		 (int)(prog_word & 0x00ff), (int)(temp & 0xff), i ))
	    }
            prog_word >>= 8; /* shift down the next byte */
	    addr += 4;  /* 1 rom address increment is 4 in VXI memory space */
	}
	buff_ptr++;
    }

cleanup: /* dumped into here by CHK() on error and ERROROUT() */
    if (vpp_on != 0)
	(void) turnOffVpp(mn);
    (void) i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, 0);
    TRACE_PRINTF(0, ("i1432_subst_zap() return %d\n", error));
    return error;
}

/***************************************************************************
 Reads the FLASH PROM in the substrate

 parms:
   mn = module pointer
   prom_addr = absolute address of F-PROM segment to be read
   size = length of block to be programmed, in bytes
   buf = pointer to block in memory to be programmed into F-PROM
 **************************************************************************/
SHORTSIZ16
i1432_subst_read_flash(E1432_MODULE_LIST_NODE *mn, ULONGSIZ32 prom_addr,
		       LONGSIZ32 size, LONGSIZ32 *buff)
{
    ULONGSIZ32 addr;
    SHORTSIZ16 error = 0;
    LONGSIZ32 fprom_id;
    LONGSIZ32 prog_word;
    LONGSIZ32* buff_ptr = buff;
    int i, j;
    LONGSIZ32 temp;
    SHORTSIZ16 page;

    TRACE_PRINTF(0, ("i1432_subst_read_flash(0x%p, 0x%lx, 0x%lx, 0x%p)\n",
		     mn, prom_addr, size, buff));

    /* program page register */
    page = E1432_FPROM_PAGE;
    if (mn->a24_256k)
    {
	page <<= 2;
	page += 2;
    }
    CHK(i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, page));

    /* check id of F-PROM to make sure it works */
    CHK(i1432_direct_write32_register_a(mn, prom_addr, E1432_A24, READ_ID));
    /* read and discard Manufactuer ID so this will work with all vendors */
    /* upper byte */
    CHK(i1432_direct_read32_register_a(mn, prom_addr, E1432_A24, &temp));
    /* read and check device ID */
    /* lower byte */
    CHK(i1432_direct_read32_register_a(mn, prom_addr+4, E1432_A24, &temp));
    fprom_id = temp & 0xFF;
    if (fprom_id != FPROM_DEVICE_CODE)
    {
        i1432_error_info = "Incorrect ROM device code.";
	error = i1432_la_print_error(mn->la, ERR1432_BAD_FLASH);
	ERROROUT(("i1432_subst_zap(): Wrong flash PROM DEVICE CODE,"
	 " expected 0x%X, got 0x%x\n", (int)FPROM_DEVICE_CODE, (int)fprom_id))
    }
    CHK(i1432_direct_write32_register_a(mn, prom_addr, E1432_A24, READ_MODE));


    /* do the reading */
    addr = prom_addr;
    for ( i = 0; i < size; i += 4 )
    {
    	prog_word = 0;
	for ( j = 0; j < 4; j++ )
	{
            CHK(i1432_direct_read32_register_a(mn, addr, E1432_A24, &temp));
            prog_word |= ((temp & 0x00ffUL) << (j * 8)); /* shift up the byte */
	    addr += 4;  /* 1 rom address increment is 4 in VXI memory space */
	}
        *buff_ptr = prog_word;
	buff_ptr++;
    }

cleanup: /* dumped into here by CHK() on error and ERROROUT() */
    (void) i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, 0);
    TRACE_PRINTF(0, ("i1432_subst_read_flash() return %d\n", error));
    return error;
}


/***************************************************************************
 *                                                                         *
 *  Sonata B EEPROM code                                                   *
 *                                                                         *
 ***************************************************************************/

/* Computes the byte address of the SCA register, assuming page map set
   for the beginning of the SCA address space. */
/* reg is a word address, relative to the base address for that SCA */
LONGSIZ32
i1432_sca_addr(E1432_MODULE_LIST_NODE *mn, SHORTSIZ16 sca, int reg)
{
    LONGSIZ32    addr;

    if ( mn->a24_256k ) addr = 0; /* only upper window mapped */
    else addr = E1432_A24_UPPER_1M;
    addr += sca * E1432_SCA_SPACING; /* Add in the sca offset */
    addr += reg << 2;	/* Add in register address, X4 for byte address */
    return addr;
}

SHORTSIZ16
i1432_write_page_reg(E1432_MODULE_LIST_NODE *mn, SHORTSIZ16 page)
{
    SHORTSIZ16 error = 0;

    /* correct for mapping, except base page */
    if (page > 0 && mn->a24_256k)
    {
	page <<= 2;
	page += 2;
    }

    CHK(i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, page));

cleanup: /* dumped into here by CHK() on error */
    return error;
}

SHORTSIZ16
i1432_sonata_check_for_err(E1432_MODULE_LIST_NODE *mn, SHORTSIZ16 sca,
  SHORTSIZ16 offset)
{
    SHORTSIZ16 error = 0;
    LONGSIZ32 imr_addr;
    LONGSIZ32 isr_addr, isr;
    LONGSIZ32 gxcr_addr, gxcr;

    imr_addr = i1432_sca_addr(mn, sca, E1433_SCA_SONATA_IMR);
    isr_addr = i1432_sca_addr(mn, sca, E1433_SCA_SONATA_ISR);
    gxcr_addr = i1432_sca_addr(mn, sca, E1433_SCA_SONATA_GXCR);

    /* Could do a i1432_direct_ping_register() here but it take 5 mS,
       should do separately, when desired */

    /* Then check that the Xilinx hasn't gotten hosed and sca bus errored */

    /* ISR read may fail, precharge the sca bus err bit to a "1" */
    /* IMR is defunct, so use it to write to */
    CHK(i1432_direct_write32_register_a(mn, imr_addr, E1432_A24,
      E1433_SCA_SONATA_ISR_BUS_ERR));
    /* Then, read the ISR */
    CHK(i1432_direct_read32_register_a(mn, isr_addr, E1432_A24, &isr));
    if ( isr & E1433_SCA_SONATA_ISR_BUS_ERR )
    {
	return i1432_print_error(ERR1432_SCA_BUS_ERROR);
    }

    /* Now, check for firmware errors, as indicated by xxx bit */
    CHK(i1432_direct_read32_register_a(mn, gxcr_addr, E1432_A24, &gxcr));
    /* Select bit for the specified DSP */
    gxcr &= E1433_SCA_SONATA_GXCR_ERR << 4 * offset;
    if ( gxcr )
    {
	/* Could query the error in the SCA here, but that would only
	   be helpful in the development phase. */
#if 0 /* Code to query the error queue */
        LONGSIZ32 cvr_addr = i1432_sca_addr(mn, sca, E1433_SCA_SONATA_CVR);
        LONGSIZ32 hdata_addr = i1432_sca_addr(mn, sca, E1433_SCA_SONATA_HDATA);
	LONGSIZ32 err_num;
	int i = 8; /* read 8 at most */
        (void) printf("  SCA FW err: sca %d, offset %d\n   ", sca, offset);
	do
	{
            CHK(i1432_direct_write32_register_a(mn, cvr_addr, E1432_A24,
	      E1433_SCA_SONATA_EE_READ_ERRS));
            i1432_pause(0.00001);  /* give the host command a while */
            CHK(i1432_direct_read32_register_a(mn, hdata_addr, E1432_A24,
	      &err_num));
	    err_num = (err_num >> 8) & 0xffffff;
	    if ( err_num != 0xffffff ) printf(" 0x%6.6x", err_num);
	}
        while ( --i > 0 && err_num != 0xffffff );
        (void) printf("\n");
#endif
        return i1432_print_error(ERR1432_SCA_FIRMWARE_ERROR);
    }

cleanup: /* dumped into here by CHK() on error */
    return error;
}

/* only sends to offset 0, the DSP that can talk to the EEPROM */
SHORTSIZ16
i1432_sonata_send_ee_cmd(E1432_MODULE_LIST_NODE *mn, SHORTSIZ16 sca,
  LONGSIZ32 cmd, LONGSIZ32 *rtn_val)
{
    SHORTSIZ16 error = 0;
    LONGSIZ32 rtn_stat;
    LONGSIZ32 cvr_addr;
    LONGSIZ32 hdata_addr;
    int j;

    CHK(i1432_sonata_check_for_err(mn, sca, 0));

    /* write the command vector */
    cvr_addr = i1432_sca_addr(mn, sca, E1433_SCA_SONATA_CVR); /* offset 0 */
    CHK(i1432_direct_write32_register_a(mn, cvr_addr, E1432_A24, cmd));

    /* Xilinx does not allow CVR to be read (to read the HC bit) so wait
       a short while instead. */
    CHK(i1432_sonata_check_for_err(mn, sca, 0)); /* waits + error checks */

    /* Wait for the EEPROM serial command output to complete. */
    hdata_addr = i1432_sca_addr(mn, sca, E1433_SCA_SONATA_HDATA);
    j = 1000; /* crude timeout */
    do
    {
        CHK(i1432_direct_write32_register_a(mn, cvr_addr, E1432_A24,
	  E1433_SCA_SONATA_EE_STAT));
        CHK(i1432_sonata_check_for_err(mn, sca, 0)); /* waits + error checks */
        CHK(i1432_direct_read32_register_a(mn, hdata_addr, E1432_A24,
	  &rtn_stat));
	/* right justify the 24 bits */
	rtn_stat = (rtn_stat >> 8) & 0xffffff;
    }
    while ( (rtn_stat & 0x800000) != 0 && j-- > 0 );

    /* Check that really completed */
    if ( rtn_stat & 0x800000 )
    {
        return i1432_print_error(ERR1432_SCA_FIRMWARE_ERROR);
    }

    /* Return the status word, which also contains the serial data read,
       if requested */
    if ( rtn_val ) *rtn_val = rtn_stat;

cleanup: /* dumped into here by CHK() on error */
    return error;
}

/* Execute a 1 parameter hostport command/vector to dsp0 on sca connector sca,
   don't wait for completion. */
SHORTSIZ16
i1432_sonata_send_hp_cmd1(E1432_MODULE_LIST_NODE *mn, SHORTSIZ16 sca,
  LONGSIZ32 parm, LONGSIZ32 cmd)
{
    SHORTSIZ16 error = 0;
    LONGSIZ32 hdata_addr = i1432_sca_addr(mn, sca, E1433_SCA_SONATA_HDATA);
    LONGSIZ32 cvr_addr = i1432_sca_addr(mn, sca, E1433_SCA_SONATA_CVR);

    /* Put the parameter into the Host Data Register... */
    CHK(i1432_direct_write32_register_a(mn, hdata_addr, E1432_A24, parm << 8));

    /* ...and then execute the hostport command/vector. */
    CHK(i1432_direct_write32_register_a(mn, cvr_addr, E1432_A24, cmd));

cleanup: /* dumped into here by CHK() on error */
    return error;
}

/* returns 1 if Sonata B, 0 if not, (negative) error number, if one */
SHORTSIZ16
i1432_is_sonata_b(SHORTSIZ16 la, SHORTSIZ16 sca)
{
    SHORTSIZ16 error = 0;
    E1432_MODULE_LIST_NODE node;
    E1432_MODULE_LIST_NODE *mn = &node;
    struct i1432_chan_info tmp_chan_list;
    struct i1432_chan_info *save_chan_list;
    int save_chan_count;
    LONGSIZ32 id_reg, id_addr;

    TRACE_PRINTF(0, ("i1432_is_sonata_b(%d, %d)\n", la, sca));
    
    /* Do this before calling i1432_fake_setup_sicl */
#if (defined(HAVE_SICL)||defined(HAVE_VTL))
    mn->sicl_id = 0;
#endif
    mn->a24_base = NULL;
    save_chan_list = i1432_chan_list[0];
    save_chan_count = i1432_chan_count[0];

    CHK(i1432_fake_setup_sicl(mn, la, &tmp_chan_list, 1));

    /* program page register */
    CHK(i1432_write_page_reg(mn, E1432_SCA_PAGE));

    id_addr = i1432_sca_addr(mn, sca, E1432_SCA_ISR);

    error = i1432_direct_ping_register(mn, id_addr);
    if ( error )
    {
	/* If we got a bus error, there's no SCA there,
	   so the SCA is certainly NOT Sonata B. */
        if ( error == ERR1432_BUS_ERROR ) error = 0;
	goto cleanup;
    }

    CHK(i1432_direct_read32_register_a(mn, id_addr, E1432_A24, &id_reg));

    error = ((id_reg & E1432_SCA_ISR_SONATA_B_FIXED_MASK) ==
      E1432_SCA_ISR_SONATA_B_FIXED_BITS_A1);

cleanup: /* dumped into here by CHK() on error */
    (void)i1432_write_page_reg(mn, E1432_BASE_PAGE);
    (void)i1432_fake_cleanup_sicl(mn, save_chan_list, save_chan_count);
    return error;
}

/* download the EEPROM i/o code to DSPs on board at sca connector */
SHORTSIZ16
i1432_download_sca_ee_io(SHORTSIZ16 la, SHORTSIZ16 sca)
{
    SHORTSIZ16 error = 0;
    E1432_MODULE_LIST_NODE node;
    E1432_MODULE_LIST_NODE *mn = &node;
    struct e1432_hwconfig hwconfig;
    struct i1432_chan_info tmp_chan_list;
    struct i1432_chan_info *save_chan_list;
    int save_chan_count;
    LONGSIZ32 *pgm;
    LONGSIZ32 gcr_addr;
    LONGSIZ32 hisr_addr, hisr;
    LONGSIZ32 hdata_addr, hdata;
    LONGSIZ32 hcr_addr;
    LONGSIZ32 isr_addr, isr;
    int i, j;

    TRACE_PRINTF(0, ("i1432_download_sca_ee_io(%d, %d)\n", la, sca));
    
    /* Do this before calling i1432_fake_setup_sicl */
#if (defined(HAVE_SICL)||defined(HAVE_VTL))
    mn->sicl_id = 0;
#endif
    mn->a24_base = NULL;
    save_chan_list = i1432_chan_list[0];
    save_chan_count = i1432_chan_count[0];

    CHK(i1432_fake_setup_sicl(mn, la, &tmp_chan_list, 1));

    /* If there is code installed in the 96k (what this tests for) and if
       a measurement is running, the SCA data DMA collides with the code
       download stuff below and this routine fails (any of several failures,
       including "Bad bits in Sonata" and "SCA FW err".  So this sends the
       RESET_MEASURE_START command, which executes MEAS_restart() in
       the 96k, which stops DMA, etc.
       e1432_meas_restart() is not used here (would be better if could)
       because hw id and module id are not available and creating ones
       would probably hose the existing ones in the calling routines.  */
    if ( ! e1432_get_hwconfig(1, &la, &hwconfig) )
    {
#ifdef	CMD_REG_WORKAROUND
        (void)i1432_direct_write32_register(mn, E1432_ALT_COMMAND_REG,
	  E1432_CMD_RESET_MEASURE_START);
        (void)i1432_direct_write32_register(mn, E1432_HOSTB_CVR_REG,
	  E1432_HOSTB_CVR_VXICMD);
#else
        (void)i1432_direct_write32_register(mn, E1432_COMMAND_REG,
	  E1432_CMD_RESET_MEASURE_START);
#endif
        i1432_pause(0.1);  /* give time to respond */
    }

    /* program page register */
    CHK(i1432_write_page_reg(mn, E1432_SCA_PAGE));

    while ( sca < 4 )
    {
	/* Only download at offset 0 - it is the one in which the EEPROM
	   bits match the downloadable. */
        gcr_addr = i1432_sca_addr(mn, sca, E1433_SCA_SONATA_GCTLR);
        hisr_addr = i1432_sca_addr(mn, sca, E1433_SCA_SONATA_HISR);
        hdata_addr = i1432_sca_addr(mn, sca, E1433_SCA_SONATA_HDATA);
        hcr_addr = i1432_sca_addr(mn, sca, E1433_SCA_SONATA_HCR);
        isr_addr = i1432_sca_addr(mn, sca, E1433_SCA_SONATA_ISR);

	/* check that this SCA is not hosed */
        CHK(i1432_direct_ping_register(mn, hisr_addr));

        /* reset DSPs, Xilinx */
        CHK(i1432_direct_write32_register_a(mn, gcr_addr, E1432_A24,
	  E1433_SCA_SONATA_GCTLR_DSP_RST));
	/* Clear (read, toss) ISR */
	(void)i1432_direct_read32_register_a(mn, isr_addr, E1432_A24, &isr);
	i1432_pause(0.001);  /* allow time for DSP to reset */
        CHK(i1432_direct_write32_register_a(mn, gcr_addr, E1432_A24, 0));

        pgm = (LONGSIZ32 *)son_ee_code;

	for ( i = 0; i < son_ee_code_size; i++ )
	{
	    j = 100; /* a crude timeout */
            do
            {
                CHK(i1432_direct_read32_register_a(mn, hisr_addr, E1432_A24,
		  &hisr));
            }
            while ( (hisr & E1433_SCA_SONATA_HISR_TXDE) == 0 && j-- > 0 );
            if ( (hisr & E1433_SCA_SONATA_HISR_TXDE) == 0 )
            {
	        error = ERR1432_SCA_HOSTPORT_FAIL;
	        (void) i1432_print_error(error);
		goto cleanup;
            }
            CHK(i1432_direct_write32_register_a(mn, hdata_addr, E1432_A24,
	      *pgm++ << 8));
	}
	/* this signals the DSP to boot up */
        CHK(i1432_direct_write32_register_a(mn, hcr_addr, E1432_A24, 0x0800));

	i1432_pause(0.0001);  /* allow time to boot up */

	/* and verify that booted up OK */
        CHK(i1432_sonata_check_for_err(mn, sca, 0));
        CHK(i1432_direct_read32_register_a(mn, hdata_addr, E1432_A24, &hdata));
        /* select the actual 24 bit data */
	hdata = (hdata >> 8) & 0xffffff;
	if ( hdata != 0x123456 )
	{
	    /* Didn't get our "magic number" back - something bombed */
	    error = ERR1432_SCA_FIRMWARE_ERROR;
            (void) i1432_print_error(error);
	    goto cleanup;
	}

	sca += 2; /* do the second sca on board the second time through */
    }

cleanup: /* dumped into here by CHK() on error */
    (void)i1432_write_page_reg(mn, E1432_BASE_PAGE);
    (void)i1432_fake_cleanup_sicl(mn, save_chan_list, save_chan_count);
    return error;
}

SHORTSIZ16
i1432_read_sca_eeprom_block(SHORTSIZ16 la, SHORTSIZ16 sca, LONGSIZ32 addr,
  LONGSIZ32 length, LONGSIZ32 *data)
{
    SHORTSIZ16 error = 0;
    E1432_MODULE_LIST_NODE node;
    E1432_MODULE_LIST_NODE *mn = &node;
    struct i1432_chan_info tmp_chan_list;
    struct i1432_chan_info *save_chan_list;
    int save_chan_count;
    LONGSIZ32 hdata_addr;
    LONGSIZ32 tmp_data;
    int i;

    TRACE_PRINTF(0, ("i1432_read_sca_eeprom(%d, %d, 0x%x, *0x%x = 0x%x)\n",
      la, sca, addr, data, *data));
    
    /* Do this before calling i1432_fake_setup_sicl */
#if (defined(HAVE_SICL)||defined(HAVE_VTL))
    mn->sicl_id = 0;
#endif
    mn->a24_base = NULL;
    save_chan_list = i1432_chan_list[0];
    save_chan_count = i1432_chan_count[0];

    CHK(i1432_fake_setup_sicl(mn, la, &tmp_chan_list, 1));

    /* program page register */
    CHK(i1432_write_page_reg(mn, E1432_SCA_PAGE));

    /* Make sure all is OK */
    CHK(i1432_sonata_check_for_err(mn, sca, 0));

    hdata_addr = i1432_sca_addr(mn, sca, E1433_SCA_SONATA_HDATA);

    addr *= 2; /* incoming address is in words, EEPROM address in half words */

    while ( length-- > 0 )
    {
	*data = 0;
       /* do as big endian - least significant half word in upper address */
	for ( i = 0; i < 2; i++ ) /* do least significant half word first */
	{
            /* Load the address */
            CHK(i1432_sonata_send_hp_cmd1(mn, sca, addr,
	      E1433_SCA_SONATA_EE_ADDR));
            /*  NOT clear why have to do this twice! */
            CHK(i1432_sonata_send_hp_cmd1(mn, sca, addr,
	      E1433_SCA_SONATA_EE_ADDR));

            /* Then, execute the read command */
            CHK(i1432_sonata_send_ee_cmd(mn, sca,
	      E1433_SCA_SONATA_EE_READ, &tmp_data));

	    /* move previous half word to upper half, put this in lower half */
	    *data = (*data << 16) | (tmp_data & 0xffff);

	    addr++;
	}
	data++;
    }

cleanup: /* dumped into here by CHK() on error */
    (void)i1432_write_page_reg(mn, E1432_BASE_PAGE);
    (void)i1432_fake_cleanup_sicl(mn, save_chan_list, save_chan_count);
    return error;
}

SHORTSIZ16
i1432_read_sca_eeprom(SHORTSIZ16 la, SHORTSIZ16 sca, LONGSIZ32 addr,
  LONGSIZ32 *data)
{
    TRACE_PRINTF(0, ("i1432_read_sca_eeprom(%d, %d, 0x%x, *0x%x = 0x%x)\n",
      la, sca, addr, data, *data));

    /* do as a block of size 1 */
    return i1432_read_sca_eeprom_block(la, sca, addr, 1, data);
}

SHORTSIZ16
i1432_write_sca_eeprom_block(SHORTSIZ16 la, SHORTSIZ16 sca, LONGSIZ32 addr,
  LONGSIZ32 length, LONGSIZ32 *data)
{
    SHORTSIZ16 error = 0;
    E1432_MODULE_LIST_NODE node;
    E1432_MODULE_LIST_NODE *mn = &node;
    struct i1432_chan_info tmp_chan_list;
    struct i1432_chan_info *save_chan_list;
    int save_chan_count;
    LONGSIZ32 hdata_addr;
    LONGSIZ32 cvr_addr;
    LONGSIZ32 tmp_data, wr_stat;
    int i, j;

    TRACE_PRINTF(0, ("i1432_write_sca_eeprom_block(%d, %d, 0x%x, 0x%x, 0x%x)\n",
      la, sca, addr, length, data));
    
    /* Do this before calling i1432_fake_setup_sicl */
#if (defined(HAVE_SICL)||defined(HAVE_VTL))
    mn->sicl_id = 0;
#endif
    mn->a24_base = NULL;
    save_chan_list = i1432_chan_list[0];
    save_chan_count = i1432_chan_count[0];

    CHK(i1432_fake_setup_sicl(mn, la, &tmp_chan_list, 1));

    /* program page register */
    CHK(i1432_write_page_reg(mn, E1432_SCA_PAGE));

    hdata_addr = i1432_sca_addr(mn, sca, E1433_SCA_SONATA_HDATA);
    cvr_addr = i1432_sca_addr(mn, sca, E1433_SCA_SONATA_CVR); /* offset 0 */

    /* Make sure all is OK */
    CHK(i1432_direct_ping_register(mn, hdata_addr));
    CHK(i1432_sonata_check_for_err(mn, sca, 0));

    /* Enable write */
    CHK(i1432_sonata_send_ee_cmd(mn, sca, E1433_SCA_SONATA_EE_WRITE_EN, 0));

    addr *= 2;  /* 2 16 bit EEPROM locations per 32 bit word coming in */
    while ( length-- > 0 )
    {
	/* Each write is 16 bits, so do both upper 16 and lower 16 bits. */
        tmp_data = (*data & 0x0000ffff);  /* least significant half first */
	for ( i = 1; i >= 0; i-- ) /* upper address first - big endian */
	{
            /* Load in the data... */
            CHK(i1432_sonata_send_hp_cmd1(mn, sca, tmp_data,
	      E1433_SCA_SONATA_EE_DATA));
            /*  NOT clear why have to do this twice! */
            CHK(i1432_sonata_send_hp_cmd1(mn, sca, tmp_data,
	      E1433_SCA_SONATA_EE_DATA));

            /* ...and the address... */
            CHK(i1432_sonata_send_hp_cmd1(mn, sca, addr + i,
	      E1433_SCA_SONATA_EE_ADDR));
            /*  NOT clear why have to do this twice! */
            CHK(i1432_sonata_send_hp_cmd1(mn, sca, addr + i,
	      E1433_SCA_SONATA_EE_ADDR));

	    /* wait until no write pending */
	    j = 1000; /* grut timeout */
	    do
	    {
		/* returns 1 if write is complete, 0 if still pending */
                CHK(i1432_sonata_send_ee_cmd(mn, sca,
		  E1433_SCA_SONATA_EE_WR_STAT, &wr_stat));
	    }
	    while ( ! wr_stat && j-- > 0 );
	    if ( ! wr_stat )
	    {
		/* timed out - error */
	        error = ERR1432_SCA_FIRMWARE_ERROR;
                (void) i1432_print_error(error);
	        goto cleanup;
	    }

            /* ...then execute the EEPROM write command. */
            CHK(i1432_sonata_send_ee_cmd(mn, sca, E1433_SCA_SONATA_EE_WRITE,
	      0));
	    
	    /* most significant half word into next (lower address) */
            tmp_data = (*data >> 16) & 0x0000ffff;
	}
	addr += 2;
	data++;
    }

    /* Disable write */
    CHK(i1432_sonata_send_ee_cmd(mn, sca, E1433_SCA_SONATA_EE_WRITE_DIS, 0));

cleanup: /* dumped into here by CHK() on error */
    (void)i1432_write_page_reg(mn, E1432_BASE_PAGE);
    (void)i1432_fake_cleanup_sicl(mn, save_chan_list, save_chan_count);
    return error;
}

SHORTSIZ16
i1432_write_sca_eeprom(SHORTSIZ16 la, SHORTSIZ16 sca, LONGSIZ32 addr,
  LONGSIZ32 data)
{
    TRACE_PRINTF(0, ("i1432_write_sca_eeprom(%d, %d, 0x%x, 0x%x)\n",
      la, sca, addr, data));

    /* do as a block of size 1 */
    return i1432_write_sca_eeprom_block(la, sca, addr, 1, &data);
}


/***************************************************************************
 Clarinet F-PROM code
 **************************************************************************/


/***************************************************************************
 Waits for the source SCA SISR_DMDACK bit to be set, in the module at
  "mn", in the source board at SCA base address  "sb".
 
 **************************************************************************/
SHORTSIZ16 
i1432_wait_src_sisr_cmdack(E1432_MODULE_LIST_NODE *mn, 
	LONGSIZ32 sb, char* msg)
{
    const int max_timeout_loops = 1000;
    SHORTSIZ16 error = 0;
    LONGSIZ32 status = 0;
    int i;

    CHK(i1432_direct_read32_register_a(mn, sb + E1432_SRC_SISR, E1432_A24, &status));
    for ( i = 0; i < max_timeout_loops && ! (status & E1432_CLARI_SISR_CMDACK);
      i++ )
    {
        CHK(i1432_direct_read32_register_a(mn, sb + E1432_SRC_SISR, E1432_A24, &status));
    }


    /* use sleep if simple, short busy loop fails */
    for ( i = 0; (i < 20 /*sec*/) && !(status & E1432_CLARI_SISR_CMDACK);
      i++ )
    {
	i1432_pause(1);
        CHK(i1432_direct_read32_register_a(mn, sb + E1432_SRC_SISR, E1432_A24, &status));
    }

    if ( ! (status & E1432_CLARI_SISR_CMDACK) )
    {
        i1432_error_info = msg;
	error = i1432_la_print_error(mn->la, ERR1432_ACK_TIMEOUT);
	goto cleanup;
    }

cleanup: /* dumped into here by CHK() on error and ERROROUT() */
    TRACE_PRINTF(4, ("i1432_wait_clarinet_sisr_cmdack() return %d\n", error));
    return error;
}


/***************************************************************************
 Handshakes a command to Clarinet host port command vector register.
 **************************************************************************/
SHORTSIZ16 
i1432_write_ack_src_hp_cmd(E1432_MODULE_LIST_NODE *mn, LONGSIZ32 sb, 
		LONGSIZ32 cmd)
{
    SHORTSIZ16 error = 0;
    LONGSIZ32 status;
    char msg[100];

    (void) sprintf(msg, "i1432_write_ack_src_hp_cmd (0x%.8lx)", cmd);
    /* to clear */
    CHK(i1432_direct_read32_register_a( mn, sb + E1432_SRC_SISR, E1432_A24, &status));
    CHK(i1432_direct_write32_register_a(mn, sb + E1432_SRC_CVR, E1432_A24, cmd)); /* do cmd */
    /* handshake */
    CHK(i1432_wait_src_sisr_cmdack(mn, sb, msg));

cleanup: /* dumped into here by CHK() on error and ERROROUT() */
    TRACE_PRINTF(4, ("i1432_write_ack_src_hp_cmd(0x%08lx) return %d\n",
      cmd, error));
    return error;
}

/***************************************************************************
 writes a command to Clarinet host port command vector register with no handshake.
 **************************************************************************/
SHORTSIZ16 
i1432_write_nack_src_hp_cmd(E1432_MODULE_LIST_NODE *mn, LONGSIZ32 sb, 
		LONGSIZ32 cmd)
{
    SHORTSIZ16 error = 0;
    LONGSIZ32 status;

    CHK(i1432_direct_read32_register_a( mn, sb + E1432_SRC_SISR, E1432_A24, &status));
    CHK(i1432_direct_write32_register_a(mn, sb + E1432_SRC_CVR, E1432_A24, cmd)); /* do cmd */

cleanup: /* dumped into here by CHK() on error and ERROROUT() */
    TRACE_PRINTF(4, ("i1432_write_nack_src_hp_cmd(0x%08lx) return %d\n",
      cmd, error));
    return error;
}


/***************************************************************************
 Read a single word from the Clarinet host port data register.
 **************************************************************************/
SHORTSIZ16 
i1432_read_src_hp_data(E1432_MODULE_LIST_NODE *mn,  LONGSIZ32 sb,
		LONGSIZ32 *word)
{
    SHORTSIZ16 error = 0;
    LONGSIZ32 temp;
    /* only upper byte is significant */
    const unsigned long byte_mask = 0xff000000UL;

    /* (void) PRINTF("i1432_read_clarinet_hp_data() reading CLAR_RTX = "); */

    *word = 0;

    CHK(i1432_direct_read32_register_a(mn, sb + E1432_SRC_RTXH, E1432_A24, &temp));
    *word |= temp & byte_mask;
    CHK(i1432_direct_read32_register_a(mn, sb + E1432_SRC_RTXM, E1432_A24, &temp));
    *word |= (temp & byte_mask) >> 8;
    CHK(i1432_direct_read32_register_a(mn, sb + E1432_SRC_RTXL, E1432_A24, &temp));
    *word |= (temp & byte_mask) >> 16;

    /* (void) PRINTF("0x%.8x\n", *word); */

cleanup: /* dumped into here by CHK() on error and ERROROUT() */
    return error;
}


/***************************************************************************
 Write a single word to the Clarinet host port data register.
 Word should be left justified: the top 3 MSBytes are sent,
  (unlike e1432_write_dsp_port() ).
 **************************************************************************/
SHORTSIZ16 
i1432_write_src_hp_data(E1432_MODULE_LIST_NODE *mn, LONGSIZ32 sb, 
			LONGSIZ32 word)
{
    SHORTSIZ16 error = 0;

    /* (void) PRINTF("i1432_write_clarinet_hp_data() writing CLAR_RTX"
	 " = 0x%.8x\n", word); */

    CHK(i1432_direct_write32_register_a(mn, sb + E1432_SRC_RTXH, E1432_A24, word));
    word <<= 8; /* shift next byte up */
    CHK(i1432_direct_write32_register_a(mn, sb + E1432_SRC_RTXM, E1432_A24, word));
    word <<= 8; /* shift next byte up */
    CHK(i1432_direct_write32_register_a(mn, sb + E1432_SRC_RTXL, E1432_A24, word));

cleanup: /* dumped into here by CHK() on error and ERROROUT() */
    return error;
}


/***************************************************************************
 Read a block of data from the Clarinet host port data register.
 **************************************************************************/
SHORTSIZ16
i1432_read_src_hp_block(E1432_MODULE_LIST_NODE *mn, LONGSIZ32 sb,
			LONGSIZ32 size, LONGSIZ32 *buff)
{
    SHORTSIZ16 error = 0;
    int i;
    LONGSIZ32 *ptr = buff;
    LONGSIZ32 block_number = E1432_CLARI_ONE;

    /* put in cmd buffer mode, srccmd */
    CHK(i1432_write_ack_src_hp_cmd(mn, sb, E1432_CLARI_MONHPDCMDBUF));
    while ( ptr - buff < size )
    {
	TRACE_PRINTF(2, ("i1432_read_src_hp_block() reading block 0x%lx\n",
	 block_number));
	/* set up for read from memory */
        CHK(i1432_write_src_hp_data(mn, sb, E1432_CLARI_MONCMD_BDPARMSRD));
	/* set block number to be downloaded */
        CHK(i1432_write_src_hp_data(mn, sb, block_number));
	/* send extended mesage */
        CHK(i1432_write_ack_src_hp_cmd(mn, sb, E1432_CLARI_MONEXTMSG));
	/* read the first, garbage word */
        CHK(i1432_read_src_hp_data(mn, sb, ptr));
	/* transfer the block of data */            
        for ( i = 0; i < E1432_CLARI_BLOCK_SIZE && ptr - buff < size; i++ )
	{
            CHK(i1432_read_src_hp_data(mn, sb, ptr++));
	}
	/* end with the host port in cmd buffer mode, srccmd */
        CHK(i1432_write_ack_src_hp_cmd(mn, sb, E1432_CLARI_MONHPDCMDBUF));
        block_number += E1432_CLARI_ONE;
    }

cleanup: /* dumped into here by CHK() on error and ERROROUT() */
    TRACE_PRINTF(2, ("i1432_read_src_hp_block() return %d\n", error));
    return error;
}


/***************************************************************************
 Write a block of data to the Clarinet host port data register.
 Use this one to write the board parameter block.
 **************************************************************************/
SHORTSIZ16
i1432_write_src_hp_block(E1432_MODULE_LIST_NODE *mn, LONGSIZ32 sb, LONGSIZ32 size,
 LONGSIZ32 *buff)
{
    SHORTSIZ16 error = 0;
    int i;
    LONGSIZ32 *ptr = buff;
    LONGSIZ32 block_number = E1432_CLARI_ONE;

    /* put in cmd buffer mode, srccmd */
    CHK(i1432_write_ack_src_hp_cmd(mn, sb, E1432_CLARI_MONHPDCMDBUF));
    while ( ptr - buff < size )
    {
	TRACE_PRINTF(2, ("i1432_write_src_hp_block() writing block"
	  "0x%lx\n", block_number));
	/* set up for write into memory */
        CHK(i1432_write_src_hp_data(mn, sb, E1432_CLARI_MONCMD_BDPARMS));
	/* set block number to be downloaded */
        CHK(i1432_write_src_hp_data(mn, sb, block_number));
	/* transfer the block of data */
        for ( i = 0; i < E1432_CLARI_BLOCK_SIZE && ptr - buff < size; i++ )
	{
            CHK(i1432_write_src_hp_data(mn, sb, *ptr++));
	}
	/* send extended mesage */
        CHK(i1432_write_ack_src_hp_cmd(mn, sb, E1432_CLARI_MONEXTMSG));
	/* end with the host port in cmd buffer mode, srccmd */
        CHK(i1432_write_ack_src_hp_cmd(mn, sb, E1432_CLARI_MONHPDCMDBUF));
        block_number += E1432_CLARI_ONE;
    }

cleanup: /* dumped into here by CHK() on error and ERROROUT() */
    TRACE_PRINTF(2, ("i1432_write_src_hp_block() return %d\n", error));
    return error;
}

/***************************************************************************
 Write a block of data to the Clarinet host port data register.
 Use this one to write a code segement when zapping code (not board parms).
 Size is 32-bit (inbound) or 24-bit (outbound) words.
 **************************************************************************/
#define BLKWRITE_SPACE_X 1
SHORTSIZ16
i1432_blkwrite_src(E1432_MODULE_LIST_NODE *mn, LONGSIZ32 sb, LONGSIZ32 size,
		  LONGSIZ32 *buff, LONGSIZ32 space, LONGSIZ32 addr)
{
    SHORTSIZ16 error = 0;
    LONGSIZ32 i;
    LONGSIZ32 *ptr = buff;

    TRACE_PRINTF(2, ("  i1432_blkwrite_src"
      "(0x%p, 0x%lx, 0x%lx, 0x%p, 0x%lx, 0x%lx)\n", mn, sb, size, buff, space, addr));
    /* put in cmd buffer mode, srccmd */
    CHK(i1432_write_ack_src_hp_cmd(mn, sb, E1432_CLARI_MONHPDCMDBUF));
    
    /* send the blkwrite command and its parms */
    CHK(i1432_write_src_hp_data(mn, sb, E1432_CLARI_MONCMD_BLKWRITE));
    CHK(i1432_write_src_hp_data(mn, sb, space << 8));
    CHK(i1432_write_src_hp_data(mn, sb, addr << 8));
    CHK(i1432_write_ack_src_hp_cmd(mn, sb, E1432_CLARI_MONEXTMSG));
    i1432_pause(0.1);   /*  probably not needed */

    /* transfer the block of data */
    for ( i = 0; i < size; i++ )
    {
        CHK(i1432_write_src_hp_data(mn, sb, *ptr++));
    }
    
    /* end with the host port in cmd buffer mode, srccmd */
    CHK(i1432_write_ack_src_hp_cmd(mn, sb, E1432_CLARI_MONHPDCMDBUF));

cleanup: /* dumped into here by CHK()  */
    return error;
}

/***************************************************************************
  Sets the page map register so the SCA slot can be 
  accessed from VXI.  Returns "sb" the sca base address.  
  Must be called again if anyting changes the page map register.
  
  > nm  identifies the vxi module
  > sca  identifies the sca slot: 0..3 are front, 4 is rear FIT slot
  < sp  SCA base address, add to register offset to get address.
  	units are vxi-relative-to-module-A24-start.
  Returns 0 if OK.
 **************************************************************************/
SHORTSIZ16
i1432_map_sca(E1432_MODULE_LIST_NODE *mn, SHORTSIZ16 sca, LONGSIZ32 *sb)
{
    SHORTSIZ16 page = 0;
    SHORTSIZ16 err = 0;
    LONGSIZ32 offset;

    switch(sca) {
	case 0:	
	    page = E1432_SCA_PAGE;  *sb = E1432_SCA0_BASE;   break;
	case 1:	
	    page = E1432_SCA_PAGE;  *sb = E1432_SCA1_BASE;   break;
	case 2:	
	    page = E1432_SCA_PAGE;  *sb = E1432_SCA2_BASE;   break;
	case 3:	
	    page = E1432_SCA_PAGE;  *sb = E1432_SCA3_BASE;   break;
	case 4:
	    page = E1432_FIT_PAGE;  *sb = E1432_FIT_BASE;    break;
	default:
	    *sb = 0;
            i1432_error_info = "Illegal parameter, SCA.";
	    return i1432_print_error(ERR1432_ILLEGAL_MODE);
    }
    offset = 0;
    if (mn->a24_256k)
    {
	page <<= 2;
    }
#ifndef HAVE_VTL
    else
	offset = E1432_WINDOW_BASE_1M;
#endif
    *sb += offset;
    err = i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, page);
    if ( err ) DIAG_PRINTF(1,("write to page map register failed\n"))
    return err;
}

/***************************************************************************
  Sets the page map register so the source module can be 
  accessed from VXI.  Returns "sb" the sca base address.  Add
  E1432_SRC_ID to this to access.  Must be called again if anyting
  else fiddles with the page map register.
  
  > nm  identifies the vxi module
  < sp  SCA base address, identifies the SCA or FIT slot
  Returns 0 if OK.
 **************************************************************************/
SHORTSIZ16
i1432_map_src(E1432_MODULE_LIST_NODE *mn, SHORTSIZ16 prom, LONGSIZ32 *sb)
{
    switch(prom) {
	case E1432_ZAP_SRC12_ID_CAL: 	/* E1434 Ch 1 & 2 Id/cal */
	case E1432_ZAP_SRC12_BOOT:	/* E1434 Ch 1 & 2 code  */
	    return i1432_map_sca(mn, 0, sb);
	    /* break; */
	case E1432_ZAP_SRC34_ID_CAL:	/* E1434 Ch 3 & 4 Id/cal */
	case E1432_ZAP_SRC34_BOOT:	/* E1434 Ch 3 & 4 code */
	    return i1432_map_sca(mn, 2, sb);
	    /* break; */
	case E1432_ZAP_CLAR_ID_CAL:	/* Clarinet in rear FIT slot */
	case E1432_ZAP_CLAR_BOOT:	
	    return i1432_map_sca(mn, 4, sb);
	    /* break; */
	default:
	    *sb = 0;
            i1432_error_info = "Illegal parameter, which_rom.";
	    return i1432_print_error(ERR1432_ILLEGAL_MODE);
    }
    /* return 0; */
}

/***************************************************************************
  Checks if clarinet/clarion is there and running.
  > nm  identifies the vxi module
  > sp  SCA base address, identifies the SCA or FIT slot
  Returns 0 if OK.
 **************************************************************************/
SHORTSIZ16
i1432_src_ok(E1432_MODULE_LIST_NODE *mn, LONGSIZ32 sb)
{
    SHORTSIZ16 error = 0;              
    LONGSIZ32 temp;
    LONGSIZ32 test_word = 0x0053c100;

    TRACE_PRINTF(2, ("i1432_src_ok(nm, 0x%p)\n", (void *)sb));

    
    /* check the clarinet hardware ID register */
    error = i1432_direct_read32_register_a(mn, sb + E1432_SRC_ID_REG,
      E1432_A24, &temp);
    if ( error )
    {
	DIAG_PRINTF(1,("  Read of source ID register bus errored.\n"));
	DIAG_PRINTF(1,("  Source missing or source FIT interface broken.\n"));
	DIAG_PRINTF(1,("  Suspect Xilinx U210 or FIT bus lines\n"))
	return error;
    }
    if (((temp & E1432_CLARI_ID_BITS) != E1432_CLARINET_ID ) &&
        ((temp & E1432_CLARI_ID_BITS) != E1432_CLARION_ID ) )
    {   
        i1432_error_info = "Bad ID from source board.";
        error = i1432_la_print_error(mn->la, ERR1432_IO);    
	DIAG_PRINTF(1,("  Expected 0x%8.8lx or 0x%8.8lx,"
	  " received 0x%8.8lx & 0x%8.8lx.\n",
	  E1432_CLARINET_ID, E1432_CLARION_ID, temp & 0xffffff00UL,
	  E1432_CLARI_ID_BITS));
	DIAG_PRINTF(1,("  Suspect Xilinx or FIT bus lines.\n"))
        return error;
    }

    /* test that Clarinet FW is up by using the echo command */
    CHK(i1432_write_ack_src_hp_cmd(mn, sb, E1432_CLARI_MONECHOMD));
    CHK(i1432_write_src_hp_data(mn, sb, test_word));
    CHK(i1432_read_src_hp_data(mn, sb, &temp)); /* first read is trash */
    CHK(i1432_read_src_hp_data(mn, sb, &temp));
    if ( temp != test_word )
    {
        i1432_error_info = "Bad host port echo from source board.";
        error = i1432_la_print_error(mn->la, ERR1432_IO);    
        return error;
    }
    
    /* turn source off */
    CHK(i1432_write_ack_src_hp_cmd(mn, sb, E1432_CLARI_MONSRCOFF));
    i1432_pause(0.1);   /*  probably not needed */
    
    /* put in cmd buffer mode, srccmd */
    CHK(i1432_write_ack_src_hp_cmd(mn, sb, E1432_CLARI_MONHPDCMDBUF));
    i1432_pause(0.1);   /*  probably not needed */
    

cleanup: /* dumped into here by CHK()  */
    if ( error )
    {
	DIAG_PRINTF(1,("  read/write to source 56k hostport failed\n"));
    }
    return error;
}


/***************************************************************************
 Performs the FLASH PROM programming in Clarinet/clarion source

 parms:
   mn = module pointer
   prom = selects which SCA and which part of rom.
   size = length of block to be programmed, in bytes
   buf = pointer to block in memory to be programmed into F-PROM
 **************************************************************************/
SHORTSIZ16
i1432_src_zap(E1432_MODULE_LIST_NODE *mn, SHORTSIZ16 prom, 
		LONGSIZ32 size, LONGSIZ32 *buff)
{
    LONGSIZ32 *test_buff =
     (LONGSIZ32 *) malloc((size_t) size * sizeof(LONGSIZ32));
    LONGSIZ32 *buff_ptr, *test_ptr;
    SHORTSIZ16 error = 0;
    int i;
    LONGSIZ32 sb;	/* SCA base address, returne by map function */

    /* print_block(size, buff, "i1432_clarinet_zap()incoming block:"); */

    CHK(i1432_map_src(mn, prom, &sb));
    CHK(i1432_src_ok(mn, sb));

    /* load block into Clarinet RAM */
    CHK(i1432_write_src_hp_block(mn, sb, size, buff));

    /* program block from RAM to F-PROM */
    /* put in cmd buffer mode, srccmd */
    CHK(i1432_write_ack_src_hp_cmd(mn, sb, E1432_CLARI_MONHPDCMDBUF));
    CHK(i1432_write_src_hp_data(mn, sb, E1432_CLARI_MONCMD_BDPARMSSTORE));
    /* passwd, not erase cmd */
    CHK(i1432_write_src_hp_data(mn, sb, E1432_CLARI_MONCMD_ERASEPARMS));
    CHK(i1432_write_ack_src_hp_cmd(mn, sb, E1432_CLARI_MONEXTMSG));

    /* verify by writing all 1's into RAM, presetting from F-PROM, reading */
    test_ptr = test_buff;
    for ( i = 0; i < size; i++ ) *test_ptr = 0xffffffffUL;
    /* copy into src RAM */
    CHK(i1432_write_src_hp_block(mn, sb, size, test_buff));
    /* suck programmed data out of F-PROM into RAM */
    CHK(i1432_write_ack_src_hp_cmd(mn, sb, E1432_CLARI_MONHPDCMDBUF));
    CHK(i1432_write_src_hp_data(mn, sb, E1432_CLARI_MONCMD_BDPARMSRESET));
    CHK(i1432_write_ack_src_hp_cmd(mn, sb, E1432_CLARI_MONEXTMSG));
    /* read back and test against original */
    CHK(i1432_read_src_hp_block(mn, sb, size, test_buff));
    test_ptr = test_buff;
    buff_ptr = buff;
    for ( i = 0; i < size; i++ )
    {
	if ( (*test_ptr && 0xffffff00UL) != (*buff_ptr && 0xffffff00UL))
	{
	    error = i1432_la_print_error(mn->la, ERR1432_BAD_FLASH);
	    ERROROUT(("i1432_src_zap(): failed to program word %d,"
	     " wrote 0x%lx, got 0x%lx\n", i, *buff_ptr, *test_ptr))
	}
	test_ptr++;
	buff_ptr++;
    }


cleanup: /* dumped into here by CHK() on error and ERROROUT() */
    (void) i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, 0);
    TRACE_PRINTF(2, ("i1432_src_zap() return %d\n", error));
    return error;
}

/***************************************************************************
 Performs the FLASH PROM read in Clarinet

 parms:
   mn = module pointer
   prom = selects which SCA and which part of rom.
   size = length of block to be read, in bytes
   buf = pointer to block in memory to be read into F-PROM
 **************************************************************************/
SHORTSIZ16
i1432_src_read_flash(E1432_MODULE_LIST_NODE *mn, SHORTSIZ16 prom, 
		LONGSIZ32 size, LONGSIZ32 *buff)
{
    SHORTSIZ16 error = 0;
    LONGSIZ32 sb;	/* SCA base address, returne by map function */

    CHK(i1432_map_src(mn, prom, &sb));
    CHK(i1432_src_ok(mn, sb));

    /* suck programmed data out of F-PROM into RAM */
    CHK(i1432_write_ack_src_hp_cmd(mn, sb, E1432_CLARI_MONHPDCMDBUF));
    CHK(i1432_write_src_hp_data(mn, sb, E1432_CLARI_MONCMD_BDPARMSRESET));
    CHK(i1432_write_ack_src_hp_cmd(mn, sb, E1432_CLARI_MONEXTMSG));
    /* read back and test against original */
    CHK(i1432_read_src_hp_block(mn, sb, size, buff));

cleanup: /* dumped into here by CHK() on error and ERROROUT() */
    (void) i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, 0);
    TRACE_PRINTF(1, ("i1432_src_read_flash() return %d\n", error));
    return error;
}


/***************************************************************************
 primary function
 **************************************************************************/


/***************************************************************************
 External function to program FLASH PROM in substrate and Clarinet

 parms:
   la = logical address
   prom = which prom (see e1432.h for #defines)
   size = length of block to be programmed, in bytes
   buf = pointer to block in memory to be programmed into F-PROM
 **************************************************************************/
SHORTSIZ16 EXPORT
e1432_zap(SHORTSIZ16 la, SHORTSIZ16 prom, LONGSIZ32 size, LONGSIZ32 *buff)
{
    ULONGSIZ32 prom_addr, offset;
    E1432_MODULE_LIST_NODE node;
    E1432_MODULE_LIST_NODE *mn = &node;
    SHORTSIZ16 error = 0;
    SHORTSIZ16 error2;
    struct i1432_chan_info tmp_chan_list;
    struct i1432_chan_info *save_chan_list;
    int save_chan_count;

    TRACE_PRINTF(0, ("e1432_zap(%d, %d, 0x%lx, 0x%p)\n",
		     la, prom, size, buff));
    /* print_id_cal(buff); */

    /* check for valid ROM type */
    switch(prom) {
	case E1432_ZAP_BOOT:
	    prom_addr = BOOT_START;
	    break;
	case E1432_ZAP_ID_CAL:
	    prom_addr = ID_CAL_START;
	    break;
	case E1432_ZAP_SUBST_PARM2:
	    prom_addr = PARM2_START;
	    break;
	case E1432_ZAP_CLAR_ID_CAL:
	case E1432_ZAP_SRC12_ID_CAL: 	/* E1434 Ch 1 & 2 Id/cal */
	case E1432_ZAP_SRC34_ID_CAL:
	    /* OK, but nothing done here */
	    break;
	default:
            i1432_error_info = "Illegal parameter, which_rom.";
	    return i1432_print_error(ERR1432_ILLEGAL_MODE);
    }

    /* Do this before calling i1432_fake_setup_sicl */
#if (defined(HAVE_SICL)||defined(HAVE_VTL))
    mn->sicl_id = 0;
#endif
    mn->a24_base = NULL;
    save_chan_list = i1432_chan_list[0];
    save_chan_count = i1432_chan_count[0];

    CHK(i1432_fake_setup_sicl(mn, la, &tmp_chan_list, 1));

    /* see if the host port is activated */
    /* is this actually necessary */
    CHK(i1432_wait32_bits_match(mn, E1432_HOSTB_ICS_REG, E1432_HOSTB_ICS_HRST,
     0, 0.01, "Host port in reset, change DIP switches?"));

    switch(prom) {
	case E1432_ZAP_BOOT:
	case E1432_ZAP_ID_CAL:
	case E1432_ZAP_SUBST_PARM2:
	    if (mn->a24_256k)
		offset = E1432_WINDOW_BASE_256K;
	    else
		offset = E1432_WINDOW_BASE_1M;
	    prom_addr &= offset - 1;
	    /* With A24_256K, we already are using the movable window,
	       so we don't need to offset the prom_addr to get to the
	       movable window. */
	    if (!mn->a24_256k)
		prom_addr += offset;
	    CHK(i1432_subst_zap(mn, prom_addr, size, buff));
	    break;
	case E1432_ZAP_CLAR_ID_CAL:
	case E1432_ZAP_SRC12_ID_CAL: 	/* E1434 Ch 1 & 2 Id/cal */
	case E1432_ZAP_SRC34_ID_CAL:
	    CHK(i1432_src_zap(mn, prom, size, buff));
	    break;
    }

cleanup: /* dumped into here by CHK() on error and ERROROUT() */
    error2 = i1432_fake_cleanup_sicl(mn, save_chan_list, save_chan_count);
    if (error2)
	error = error2;
    TRACE_PRINTF(0, ("e1432_zap() return %d\n", error));
    return error;
}

/***************************************************************************
 External function to read FLASH PROM in substrate and Clarinet

 parms:
   la = logical address
   prom = which prom (see e1432.h for #defines)
   size = length of block to be read, in bytes
   buf = pointer to block in memory to be read from F-PROM
 **************************************************************************/

SHORTSIZ16 EXPORT
e1432_read_flash(SHORTSIZ16 la, SHORTSIZ16 prom, LONGSIZ32 size, LONGSIZ32 *buff)
{
    ULONGSIZ32 prom_addr, offset;
    E1432_MODULE_LIST_NODE node;
    E1432_MODULE_LIST_NODE *mn = &node;
    SHORTSIZ16 error = 0;
    SHORTSIZ16 error2;
    struct i1432_chan_info tmp_chan_list;
    struct i1432_chan_info *save_chan_list;
    int save_chan_count;

    TRACE_PRINTF(0, ("e1432_read_flash(%d, %d, 0x%lx, 0x%p)\n",
		     la, prom, size, buff));
    /* print_id_cal(buff); */

    /* Do this before calling i1432_fake_setup_sicl */
#if (defined(HAVE_SICL)||defined(HAVE_VTL))
    mn->sicl_id = 0;
#endif
    mn->a24_base = NULL;
    save_chan_list = i1432_chan_list[0];
    save_chan_count = i1432_chan_count[0];

    CHK(i1432_fake_setup_sicl(mn, la, &tmp_chan_list, 1));

    /* see if the host port is activated */
    /* is this actually necessary */
#ifndef	HPVXI_DOWNLOAD	/* in E1406, this can fail but no actual problem */
    CHK(i1432_wait32_bits_match(mn, E1432_HOSTB_ICS_REG, E1432_HOSTB_ICS_HRST,
     0, 0.01, "Host port in reset, change DIP switches?"));
#endif	/* HPVXI_DOWNLOAD */

    switch(prom) {
	case E1432_ZAP_ID_CAL:
	case E1432_ZAP_SUBST_PARM2:
	    prom_addr = (prom == E1432_ZAP_ID_CAL) ? ID_CAL_START : PARM2_START;
	    if (mn->a24_256k)
		offset = E1432_WINDOW_BASE_256K;
	    else
		offset = E1432_WINDOW_BASE_1M;
	    prom_addr &= offset - 1;
	    /* With A24_256K, we already are using the movable window,
	       so we don't need to offset the prom_addr to get to the
	       movable window. */
	    if (!mn->a24_256k)
		prom_addr += offset;
	    CHK(i1432_subst_read_flash(mn, prom_addr, size, buff));
	    break;
	case E1432_ZAP_CLAR_ID_CAL:
	case E1432_ZAP_SRC12_ID_CAL: 	/* E1434 Ch 1 & 2 Id/cal */
	case E1432_ZAP_SRC34_ID_CAL:
	    CHK(i1432_src_read_flash(mn, prom, size, buff));
	    break;
	default:
            i1432_error_info = "Illegal parameter, which_rom.";
	    return i1432_print_error(ERR1432_ILLEGAL_MODE);
    }

cleanup: /* dumped into here by CHK() on error and ERROROUT() */
    if (error)
	(void) i1432_la_print_error(mn->la, error);
    error2 = i1432_fake_cleanup_sicl(mn, save_chan_list, save_chan_count);
    if (error2)
	error = error2;
    TRACE_PRINTF(0, ("e1432_read_flash() return %d\n", error));
    return error;
}

/***************************************************************************

 Puts one segement of code in the source board FLASH PROM.  
 
 Before calling this for each segment, setup the module 
 node and erase the entire ROM. 
 
 parms:
   > mn		module list node
   > sb		SCA base address.  selects source board
   > file 	file name string. 
   > segAddr	Clarinet address.  0x00c000 for monboot, 0x00b000 for 
   			bdparmem....   Sent as a parameter for the 
   			monflashprog command.
   > progCmd	0x1E0001 = MONCMD_PROGSYS, 0x1E0002 =MONCMD_PROGPARMS

 **************************************************************************/

#ifndef HPVXI_DOWNLOAD  /* no file reads available in E1406 */
SHORTSIZ16
i1432_zap_src_segment(E1432_MODULE_LIST_NODE *mn, LONGSIZ32 sb,
		const char *fileName, 
		LONGSIZ32 segAddr, LONGSIZ32 progCmd )
{
    off_t   file_size = 0;
    int     file_des = -1;
    const LONGSIZ32 TEMP_RAM_ADDR = 0x008000UL;
    char   *buff = NULL;
    unsigned long *p;
    LONGSIZ32 i, numWords;
    int error;
           
    TRACE_PRINTF(1, ("i1432_zap_src_segment(0x%p, 0x%08lx, %s, 0x%08lx, 0x%08lx)\n",
		     mn, sb, fileName, segAddr, progCmd));

    /* open the code file */
    error = i1432_open_file(fileName, &file_des, &file_size);
    if (error) return error;

#ifdef	HAVE_SEGMENTS
    if (file_size > 65530UL) 
    {
        i1432_error_info = "File >65K";
	error = i1432_print_error(ERR1432_FILE);
	goto cleanup;
    }
#endif               

	/* allocate buffer for code */
	buff = (char *) malloc((size_t) file_size);
	if (buff == NULL)
	    error = i1432_print_error(ERR1432_MALLOC);
	if (error)
	    goto cleanup;

	/* read data from file to buf */
	if ((size_t) read(file_des, buff, (size_t) file_size) !=
	    (size_t) file_size)
	    error = i1432_print_error(ERR1432_FILE);
	if (error)
	    goto cleanup;

    numWords = file_size / 4;
    if (numWords * 4 != file_size) {
        i1432_error_info = "Odd file size.";
        error = i1432_print_error(ERR1432_FILE);    
        goto cleanup;              
    }

#if SWAP 
    error = ibeswap( buff, file_size, 4);
    if (error)
	error = i1432_print_error(ERR1432_IO);
    if (error)
	goto cleanup;
#endif              
     
     /* shift data up one byte.  File doesn't use MSByte, 
      * i1432_blkwrite_clarinet() doesn't use LSByte
      */
     p = (unsigned long *) (void *)buff;
     for (i=0; i < numWords; i++) {
     	(*p) <<= 8;
     	p++;
     }
                
     /* write the code to a RAM address in Clarinet */
     error = i1432_blkwrite_src(mn, sb, numWords, (LONGSIZ32 *)(void *)buff, 
		BLKWRITE_SPACE_X, TEMP_RAM_ADDR);        
    /* send the programming command and its parms */
    CHK(i1432_write_src_hp_data(mn, sb, progCmd));
    CHK(i1432_write_src_hp_data(mn, sb, BLKWRITE_SPACE_X << 8));
    CHK(i1432_write_src_hp_data(mn, sb, TEMP_RAM_ADDR << 8));
    CHK(i1432_write_src_hp_data(mn, sb, segAddr << 8));
    CHK(i1432_write_src_hp_data(mn, sb, numWords << 8));
    CHK(i1432_write_ack_src_hp_cmd(mn, sb, E1432_CLARI_MONFLASHPROG ));

    /* end with host port in cmd buffer mode  */
    CHK(i1432_write_ack_src_hp_cmd(mn, sb, E1432_CLARI_MONHPDCMDBUF));
        
cleanup:
    if (buff != NULL)
	free(buff);
#ifndef HPVXI_DOWNLOAD
    if (file_des != -1)
	(void) close(file_des);
#endif /* HPVXI_DOWNLOAD */

    return error;
}
#endif /* HPVXI_DOWNLOAD */

/***************************************************************************

 Erases the entire ROM and then  puts all segements of code in 
 the source-board FLASH PROM.  Gets code from individual files found
 in fileDir.

 parms:
   > la 	logical address
   > prom 	which prom, which SCA
   > doParms 	Set <>0 to overwrite cal parameters with initial values
   > fileDir 	Directory string to code files 

 **************************************************************************/

/* these segment address are sent to Clarinet to tell it where to put
 * the data 
 */

/*
defines now in e1432i.h referenced by sema.h
*/

#ifndef HPVXI_DOWNLOAD  /* no file reads available in E1406 */
SHORTSIZ16 EXPORT
e1432_zap_src_boot(SHORTSIZ16 la, SHORTSIZ16 prom, SHORTSIZ16 doParms, 
		const char *fileDir )
{
    E1432_MODULE_LIST_NODE node;
    E1432_MODULE_LIST_NODE *mn = &node;
    struct i1432_chan_info tmp_chan_list;
    struct i1432_chan_info *save_chan_list;
    int save_chan_count;
    int error;
    SHORTSIZ16 error2;
    LONGSIZ32 sb;	/* SCA base address, returned by map function */
    char fileName[100];

           
    TRACE_PRINTF(0, ("e1432_zap_src_boot(%d, %d, %d, %s)\n", la, prom, doParms, fileDir));

    /* Do this before calling i1432_fake_setup_sicl */
#if (defined(HAVE_SICL)||defined(HAVE_VTL))
    mn->sicl_id = 0;
#endif
    mn->a24_base = NULL;
    save_chan_list = i1432_chan_list[0];
    save_chan_count = i1432_chan_count[0];
    error = i1432_fake_setup_sicl(mn, la, &tmp_chan_list, 1);
    if (error) return error;
    
    fileName[100-1] = 0;
    
    CHK(i1432_map_src(mn, prom, &sb));
    CHK(i1432_src_ok(mn, sb));
    
    /* erase the code block */
    CHK(i1432_write_src_hp_data(mn, sb, E1432_CLARI_MONCMD_ERASESYS)); 
    CHK(i1432_write_ack_src_hp_cmd(mn, sb, E1432_CLARI_MONERASE));

    /* do the first monboot segment */
    (void) sprintf( fileName, "%s/%s", fileDir, "monboot.bin" );
    error = i1432_zap_src_segment(mn, sb, fileName, 
        	E1432_SRC_MONBOOT_SEGADDR, E1432_CLARI_MONCMD_PROGSYS);
    if (error) goto cleanup;

    /* do other segments */
    (void) sprintf( fileName, "%s/%s", fileDir, "monvects.bin" );
    error = i1432_zap_src_segment(mn, sb, fileName, 
        	E1432_SRC_MONVECTS_SEGADDR, E1432_CLARI_MONCMD_PROGSYS);
    if (error) goto cleanup;

    (void) sprintf( fileName, "%s/%s", fileDir, "segtbl.bin" );
    error = i1432_zap_src_segment(mn, sb, fileName, 
        	E1432_SRC_SEGTBL_SEGADDR, E1432_CLARI_MONCMD_PROGSYS);
    if (error) goto cleanup;
    (void) sprintf( fileName, "%s/%s", fileDir, "main0.bin" );
    error = i1432_zap_src_segment(mn, sb, fileName, 
        	E1432_SRC_MAIN0_SEGADDR, E1432_CLARI_MONCMD_PROGSYS);
    if (error) goto cleanup;

    (void) sprintf( fileName, "%s/%s", fileDir, "mainy0.bin" );
    error = i1432_zap_src_segment(mn, sb, fileName, 
        	E1432_SRC_MAINY0_SEGADDR, E1432_CLARI_MONCMD_PROGSYS);
    if (error) goto cleanup;

    (void) sprintf( fileName, "%s/%s", fileDir, "sine2.bin" );
    error = i1432_zap_src_segment(mn, sb, fileName, 
        	E1432_SRC_SINE2_SEGADDR, E1432_CLARI_MONCMD_PROGSYS);
    if (error) goto cleanup;

    (void) sprintf( fileName, "%s/%s", fileDir, "randbb2.bin" );
    error = i1432_zap_src_segment(mn, sb, fileName, 
        	E1432_SRC_RANDBB2_SEGADDR, E1432_CLARI_MONCMD_PROGSYS);
    if (error) goto cleanup;

    (void) sprintf( fileName, "%s/%s", fileDir, "randz2.bin" );
    error = i1432_zap_src_segment(mn, sb, fileName, 
        	E1432_SRC_RANDZ2_SEGADDR, E1432_CLARI_MONCMD_PROGSYS);
    if (error) goto cleanup;

    (void) sprintf( fileName, "%s/%s", fileDir, "arbbb2.bin" );
    error = i1432_zap_src_segment(mn, sb, fileName, 
        	E1432_SRC_ARBBB2_SEGADDR, E1432_CLARI_MONCMD_PROGSYS);
    if (error) goto cleanup;

    (void) sprintf( fileName, "%s/%s", fileDir, "gauss.bin" );
    error = i1432_zap_src_segment(mn, sb, fileName, 
        	E1432_SRC_GAUSSD_SEGADDR, E1432_CLARI_MONCMD_PROGSYS);
    if (error) goto cleanup;


    if (doParms) {
        /* program the cal parameter area with intial values */

        /* first re-boot so it knows how to program cal stuff */

	/* Sending E1432_CLARI_SRCBOOT command now will kill Clarinet
	 * because of a bug in Clarinet firmware.  It's page map register is
	 * not zeroed by the boot command.  Workaround is to re-write
	 * something to the 0 page.  */        
	(void) sprintf( fileName, "%s/%s", fileDir, "monboot.bin" );
	error = i1432_zap_src_segment(mn, sb, fileName, 
		E1432_SRC_MONBOOT_SEGADDR, E1432_CLARI_MONCMD_PROGSYS);
	if (error) goto cleanup;
	
    	i1432_pause(0.1);   
	CHK(i1432_direct_write32_register_a(mn, sb + E1432_SRC_CVR,
	  E1432_A24, E1432_CLARI_SRCBOOT)); 
    	i1432_pause(2.0);   /*  wait for re-boot */
    	CHK(i1432_src_ok(mn, sb));
    	
	/* passwd, not erase cmd */
    	CHK(i1432_write_src_hp_data(mn, sb, E1432_CLARI_MONCMD_ERASEPARMS));
    	CHK(i1432_write_ack_src_hp_cmd(mn, sb, E1432_CLARI_MONERASE));
	    
        (void) sprintf( fileName, "%s/%s", fileDir, "bdparmem.bin" );
        
        error = i1432_zap_src_segment(mn, sb, fileName, 
        	E1432_SRC_BDPARMS_SEGADDR, E1432_CLARI_MONCMD_PROGPARMS);
        if (error) goto cleanup;
    }    
    
cleanup: 
    (void) i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, 0);
    error2 = i1432_fake_cleanup_sicl(mn, save_chan_list, save_chan_count);
    if (error2)
	error = error2;
    return error;
}
#endif /* HPVXI_DOWNLOAD */


SHORTSIZ16 EXPORT
i1432_read_serial(SHORTSIZ16 la, char *serial)
{
#define BUFFLEN (E1432_SERIAL_LENGTH/4 + 1) /* just rev, serial */
#   if	SWAP	/* Strings may need fixing up. */
    SHORTSIZ16 error = 0;
#   endif	/* SWAP */
    SHORTSIZ16 err;
    LONGSIZ32 buff[BUFFLEN];
    LONGSIZ32 *rev = (LONGSIZ32 *)&buff[0];
    char *ser = (char *)&buff[1];

    err = e1432_read_flash(la, E1432_ZAP_ID_CAL, 4*BUFFLEN, buff);
    if ( err != 0 )
    {
	/* make serial null string if the read failed */
	*serial = 0;
	return err;
    }

    /* check for valid FLASH */
    /* revs > 2 program first rev length with E1432_ID_CAL_R2_REV_LENGTH */
    if ( ! VALID_ID_CAL_REV(*rev) )
    {
	*serial = 0;
	return 0;
    }

#   if	SWAP	/* Strings may need fixing up. */
    if (ibeswap(ser, E1432_SERIAL_LENGTH, 4) != 0)
    {
	*serial = 0;
	return i1432_la_print_error(la, ERR1432_SICL_ERROR);
    }
#   endif	/* SWAP */

    (void) strncpy(serial, ser, E1432_SERIAL_LENGTH);

    /* just to be sure terminated */
    serial[E1432_SERIAL_LENGTH] = 0;

    return 0;
}


SHORTSIZ16
i1432_write_serial(SHORTSIZ16 la, char *serial)
{
    SHORTSIZ16 err;
    E1432_ID_CAL_LATEST *id_cal;
    char *ser, *ctmp;
    int ser_len;
    int i;
    DIAG_PRINTF(3, ("i1432_write_serial(%d, \"%s\")\n", (int)la, serial));

    id_cal = (E1432_ID_CAL_LATEST *) malloc(sizeof(E1432_ID_CAL_LATEST));
    if ( id_cal == NULL )
    {
	return ERR1432_MALLOC;
    }

    ser_len = strlen(serial);
    if ( ser_len > E1432_SERIAL_LENGTH ) ser_len = E1432_SERIAL_LENGTH;
    ser = (char *)id_cal->serial;

    err = e1432_read_flash(la, E1432_ZAP_ID_CAL, sizeof(E1432_ID_CAL_LATEST),
      (LONGSIZ32 *)id_cal);
    if ( err != 0 ) goto cleanup;
    DIAG_PRINTF(2, ("previous serial = \"%s\"\n", ser));
    
    /* Module number, E14xx, is top byte. */
    /* Prevent rewriting E1433 serial number - option 1D1 piracy loophole */
    if ( ((id_cal->sca_info.sca[0].board_num >> 24) & 0xff) == 33 )
    {
        err = ERR1432_SERIAL_NUM_ZAP_NOT_ALLOWED;
	goto cleanup;
    }

    /* fill with zeros first */
    ctmp = ser;
    for ( i = 0; i < E1432_SERIAL_LENGTH; i++ ) *ctmp++ = '\0';

    (void) strncpy( ser, serial, ser_len);
#if SWAP	/* untested */
    err = ibeswap( serial, E1432_SERIAL_LENGTH, 4);
    if ( err )
    {
        err = ERR1432_IO;
	goto cleanup;
    }
#endif		/* SWAP */

    /* zap it out */
    err = e1432_zap(la, E1432_ZAP_ID_CAL, sizeof(E1432_ID_CAL_LATEST),
     (LONGSIZ32 *)id_cal);
    if ( err != 0 ) goto cleanup;

cleanup:
    free(id_cal);
    DIAG_PRINTF(3,
      ("i1432_write_serial(%d, \"%s\") returning %d\n", la, serial, err));
    return err;
}


#ifndef	HPVXI_DOWNLOAD	/* not needed */
static SHORTSIZ16
i1432_opt_str2num(char *optStr, LONGSIZ32 *optNum)
{
    int len = strlen(optStr);
    int aString = 0; /* until proven otherwise */
    char c;
    int i, byte;
    char   *p;

    *optNum = 0;  /* fill with '\0' */

    /* attempt to copy into optNum if a valid option string */
    if ( len <= VALID_OPT_STR_LEN_MAX && len >= VALID_OPT_STR_LEN_MIN )
    {
        aString = 1; /* until proven otherwise */
	for ( i = 0; i < len && aString; i++ )
	{
	    c = optStr[i];
	    if ( VALID_OPT_CHAR(c) )
	    {
	        byte = BYTE_SIZE_OF_LONG - 1 - i ;
		*optNum |= c << 8 * byte;
	    }
	    else
	    {
		aString = 0;
	    }
	}
	if ( aString )
	{
	    /* converted it as we checked, so return */
	    return 0;
	}
    }

#if 0
    {
	int     count;

	/* Original code.  This worked fine, except for some strange
	   problem when compiling the host library for 16-bit windows
	   for the Production test system.  How can it not like
	   sscanf?  It's ANSI/ISO C.  Somebody needs to mail a copy of
	   K&R to Bill. */
	count = sscanf(optStr, "%x", optNum);
	if ( count != 1 )
	{
	    *optNum = 0;
	    return ERR1432_INVALID_OPTION_STRING;
	}
    }
#endif

    /* New version, should be functionally the same as the #ifed out
       code above.  Attempt to convert as a hex string. */
    *optNum = strtoul(optStr, &p, 16);
    if (p == optStr)
    {
	*optNum = 0;
        return ERR1432_INVALID_OPTION_STRING;
    }

    return 0;
}
#endif	/* HPVXI_DOWNLOAD */


/***************************************************************************

 parms:
   > la 	logical address
   > oper 	E1432_PROG_CLEAR	clear all options from list
	 	E1432_PROG_DELETE	delete option from list
	 	E1432_PROG_ADD		add option to list
   > option 	option string, either plain ascii option string(4 chars max)
		or hex sprintf representation of a 4 byte word.

 **************************************************************************/

#ifndef	HPVXI_DOWNLOAD	/* not do this in E1406 download environment */
SHORTSIZ16 EXPORT
i1432_zap_parm2_opts(SHORTSIZ16 la, SHORTSIZ16 oper, char *option)
{
    SHORTSIZ16 err;
    LONGSIZ32 option32 = 0;
    LONGSIZ32 buff[E1432_FPROM_PARM_BLOCKSIZE];
    LONGSIZ32 *rev = &buff[E1432_PARM2_REV];
    LONGSIZ32 *num_opts = &buff[E1432_PARM2_NUM_OPTS];
    LONGSIZ32 *opts = &buff[E1432_PARM2_OPT_LIST];
    int i, opt_num;

    TRACE_PRINTF(1, ("i1432_zap_parm2_opts(%d, 0x%x, %s)\n", la, oper, option));

    if ( oper == E1432_PROG_DELETE || oper == E1432_PROG_ADD )
    {
        /* convert option string to 4 byte word */
        err = i1432_opt_str2num(option, &option32);
        if ( err )
        {
	    return err;
        }
    }

    /* read the existing parm2 block in */
    err = e1432_read_flash(la, E1432_ZAP_SUBST_PARM2,
      E1432_FPROM_PARM_BLOCKSIZE, buff);
    if ( err )
    {
	return err;
    }

    if ( ! VALID_PARM2_REV(*rev) )
    {
	/* initialize with current revision and 0's otherwise */
	*rev = E1432_PARM2_R_CURR;
	for ( i = 1; i < E1432_FPROM_PARM_BLOCKSIZE; i++ )
	{
	    buff[i] = 0;
	}
    }

    if ( oper == E1432_PROG_DELETE )
    {
	/* find and delete the option */
	for ( opt_num = 0; opt_num < *num_opts; opt_num++ )
	{
	    if ( *(opts + opt_num) == option32 )
	    {
                for ( i = opt_num; i < *num_opts - 1; i++ )
		{
		    *(opts + i) = *(opts + i + 1);
		}
		*(opts + --(*num_opts)) = 0;
	    }
	}
    }

    else if ( oper == E1432_PROG_ADD )
    {
        *(opts + (*num_opts)++) = option32;
    }

    else if ( oper == E1432_PROG_CLEAR )
    {
	 while ( *num_opts > 0 )
	 {
	     *(opts + --(*num_opts)) = 0;
	 }
    }

    /* Write block to hardware */
    err = e1432_zap(la, E1432_ZAP_SUBST_PARM2, E1432_FPROM_PARM_BLOCKSIZE,
      buff);

    return err;
}
#endif	/* HPVXI_DOWNLOAD */


/* Convert optNum into string optStr.
   Ascii copy if optNum contains a valid ascii option string (bytes are
   either 0, A-Z, or 0-9 and length is 3 or 4 characters).
   Otherwise fprintf as hex */
static void
i1432_opt_num2str(LONGSIZ32 optNum, char *optStr)
{
    char c;
    int i, byte;
    int aString = 1; /* until proven otherwise */

    for ( i = 0; i < BYTE_SIZE_OF_LONG && aString; i++ )
    {
	byte = BYTE_SIZE_OF_LONG - 1 - i ;
	c = (char) (optNum >> (8 * byte));
	if ( VALID_OPT_CHAR(c) )
	{
	    optStr[i] = c;
	    if ( c == '\0' && i < VALID_OPT_STR_LEN_MIN )
	    {
		aString = 0;
	    }
	}
	else
	{
	    aString = 0;
	}
    }
    if ( aString )
    {
	/* be sure it's terminated */
	optStr[BYTE_SIZE_OF_LONG] = '\0';
	return;
    }

    /* not a valid option string, use it's hex representation */
    (void) sprintf(optStr, "%x", optNum);
    return;
}


/***************************************************************************

 parms:
   > la 	logical address
   > options 	pointer to array of pointers to strings, 9 characters
		minimum each ("DDDDDDDD"), to receive the option list
   < options	list of options read, plain ascii or hex sprintf of binary
   > *max_opts 	maximum number of options to be read into list of options
   < *max_opts	number of options read into list of options

 **************************************************************************/

SHORTSIZ16 EXPORT
i1432_read_parm2_opts(SHORTSIZ16 la, char **options, SHORTSIZ16 *max_opts)
{
    LONGSIZ32 buff[E1432_FPROM_PARM_BLOCKSIZE];
    LONGSIZ32 *rev = &buff[E1432_PARM2_REV];
    LONGSIZ32 *num_opts = &buff[E1432_PARM2_NUM_OPTS];
    LONGSIZ32 *opts = &buff[E1432_PARM2_OPT_LIST];
    LONGSIZ32 nopts;
    SHORTSIZ16 err;

    /* read the parm2 block in */
    err = e1432_read_flash(la, E1432_ZAP_SUBST_PARM2,
      E1432_FPROM_PARM_BLOCKSIZE, buff);
    if ( err )
    {
	return err;
    }

    if ( ! VALID_PARM2_REV(*rev) )
    {
        return ERR1432_FLASH_INVALID;
    }

    nopts = *num_opts;
    if ( nopts > *max_opts ) nopts = *max_opts;
    *max_opts = 0;

    while( *max_opts < nopts )
    {
        i1432_opt_num2str(*opts, options[*max_opts]);
	(*max_opts)++;
	opts++;
    }

    return 0;
}
